import { json } from '@sveltejs/kit';
import { db } from '$lib/server/db.js';

/** @type {import('./$types').RequestHandler} */
export async function GET({ params }) {
  try {
    const id = parseInt(params.id);
    const [teachers] = await db.query('SELECT * FROM teachers WHERE id = ?', [id]);
    
    if (teachers.length === 0) {
      return json({ error: 'Guru tidak ditemukan' }, { status: 404 });
    }
    
    return json({ success: true, data: teachers[0] });
  } catch (error) {
    console.error('Get teacher error:', error);
    return json({ error: 'Gagal mengambil data guru' }, { status: 500 });
  }
}

/** @type {import('./$types').RequestHandler} */
export async function PUT({ params, request }) {
  try {
    const id = parseInt(params.id);
    const data = await request.json();
    const { nip, fullName, email, phone, subjects, class: className, status } = data;

    if (!fullName || !email) {
      return json({ error: 'Nama dan Email wajib diisi' }, { status: 400 });
    }

    // Convert subjects array to JSON string if it's an array
    let subjectsValue = subjects;
    if (Array.isArray(subjects)) {
      subjectsValue = JSON.stringify(subjects);
    } else if (typeof subjects === 'string' && subjects.trim() !== '') {
      // If it's a comma-separated string, convert to array then JSON
      const subjectsArray = subjects.split(',').map(s => s.trim()).filter(Boolean);
      subjectsValue = JSON.stringify(subjectsArray);
    } else {
      subjectsValue = '[]';
    }

    await db.query(
      `UPDATE teachers 
       SET nip = ?, fullName = ?, email = ?, phone = ?, subjects = ?, class = ?, status = ?
       WHERE id = ?`,
      [
        nip || null,
        fullName.trim(),
        email.trim(),
        phone || null,
        subjectsValue,
        className || null,
        status || 'aktif',
        id
      ]
    );

    // Also update users table if email matches
    await db.query(
      `UPDATE users 
       SET fullName = ?, email = ?, phone = ?
       WHERE email = ? AND role = 'guru'`,
      [fullName.trim(), email.trim(), phone || null, email.trim()]
    );

    return json({ success: true, message: 'Data guru berhasil diupdate' });
  } catch (error) {
    console.error('Update teacher error:', error);
    return json({ error: 'Gagal mengupdate data guru' }, { status: 500 });
  }
}

/** @type {import('./$types').RequestHandler} */
export async function DELETE({ params, url }) {
  try {
    const id = parseInt(params.id);
    const action = (url.searchParams.get('action') || 'permanent').toLowerCase(); // permanent | nonaktif

    const [teachers] = await db.query('SELECT email FROM teachers WHERE id = ?', [id]);
    if (teachers.length === 0) {
      return json({ error: 'Guru tidak ditemukan' }, { status: 404 });
    }
    const email = teachers[0].email;

    if (action === 'nonaktif') {
      // Nonaktifkan sementara: update status di MySQL, akun tidak bisa login
      await db.query('UPDATE teachers SET status = ? WHERE id = ?', ['nonaktif', id]);
      await db.query(
        'UPDATE users SET status = ? WHERE email = ? AND role = ?',
        ['nonaktif', email, 'guru']
      );
      return json({
        success: true,
        message: 'Guru berhasil dinonaktifkan. Akun tidak dapat login lagi.'
      });
    }

    // action === 'permanent': hapus dari database
    await db.query('DELETE FROM users WHERE email = ? AND role = ?', [email, 'guru']);
    await db.query('DELETE FROM teachers WHERE id = ?', [id]);

    return json({ success: true, message: 'Guru berhasil dihapus permanen dari database.' });
  } catch (error) {
    console.error('Delete teacher error:', error);
    return json({ error: 'Gagal menghapus guru' }, { status: 500 });
  }
}
